/* Copyright (c) 2017-2018 VMware, Inc. All rights reserved. -- VMware Confidential */
import {FileMigrator, SubContentTransformation} from "./fileMigrator";
import {TransformationInfo} from "../transformationInfo";

export class ExtensionsMigrator extends FileMigrator {
   static readonly dialogWithTitle: RegExp = /<delegate>[\s\S]*?<className>com\.vmware\.vsphere\.client\.htmlbridge\.HtmlActionDelegate<\/className>[\s\S]*?<\/delegate>/g;
   static readonly bridgeDelegate: string = "com.vmware.vsphere.client.htmlbridge.HtmlActionDelegate";

   constructor(filePath: string, fileContent: string) {
      super(filePath, fileContent);
   }

   migrateFile(transformationInfo: TransformationInfo) {
      const matches = this.getContent().match(ExtensionsMigrator.dialogWithTitle);

      matches.forEach((x) => {
         const newDelegate = this.getClassNameDelegate(x);
         const replacedDelegate = x.replace(ExtensionsMigrator.bridgeDelegate, newDelegate);
         this.setContent(this.getContent().replace(x, replacedDelegate));
      });
   }

   getAllFileTransformations(transformationInfo: TransformationInfo): SubContentTransformation[] {
      const matches = this.getContent().match(ExtensionsMigrator.dialogWithTitle);

      let result: SubContentTransformation[] = [];
      let indexOfLastModification: number = 0;

      matches.forEach((x) => {
         indexOfLastModification = this.getContent().indexOf(x, indexOfLastModification) + 1;
         const newDelegate = this.getClassNameDelegate(x);

         const indexOfDelegate = this.getContent().indexOf(ExtensionsMigrator.bridgeDelegate, indexOfLastModification);
         const delegatePrefix = this.getContent().substr(0, indexOfDelegate);
         result.push({
            oldContent: ExtensionsMigrator.bridgeDelegate,
            newContent: newDelegate,
            position: indexOfLastModification,
            errorMsg: null,
            line: ExtensionsMigrator.countNewLines(delegatePrefix),
            character: ExtensionsMigrator.countCharactersFromLastNewLine(delegatePrefix)
         });
      });

      return result;
   }

   private getClassNameDelegate(content: string): string {
      const htmlHeadlesAction: string = "com.vmware.vsphere.client.HtmlPluginHeadlessAction";
      const htmlModalAction: string = "com.vmware.vsphere.client.HtmlPluginModalAction";
      return (content.indexOf("dialogTitle") < 0) ? htmlHeadlesAction : htmlModalAction;
   }
}